
/**
 * YouTube Playlist List Extractor to CSV
 * Developed by DigiMoor.com
 * 
 * Security Note for Reviewers:
 * - This extension runs purely locally.
 * - No data is sent to external servers.
 * - 'scripting' permission is used to scroll the page and read DOM elements.
 * - 'downloads' permission is used to save the generated CSV.
 */

let allPlaylists = [];
let allVideos = [];
let isScraping = false;
let userName = "User";

// Helpers
const wait = (ms) => new Promise(r => setTimeout(r, ms));

function updateStatus(msg) {
  chrome.storage.local.set({ progressMessage: msg });
}

// Listen for start command from popup
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (msg.action === "START_PROCESS") {
    startExtraction(msg.tabId, msg.url);
  }
});

async function startExtraction(startTabId, currentUrl) {
  isScraping = true;
  chrome.storage.local.set({ isScraping: true });
  allPlaylists = [];
  allVideos = [];
  userName = "User";
  
  try {
    // 1. Ensure we are on the playlist feed
    if (!currentUrl.includes("/feed/playlists")) {
      updateStatus("Navigating to Playlist Feed...");
      await chrome.tabs.update(startTabId, { url: "https://www.youtube.com/feed/playlists" });
      
      // Wait for load
      await new Promise(resolve => {
        const listener = (tabId, info) => {
          if (tabId === startTabId && info.status === 'complete') {
            chrome.tabs.onUpdated.removeListener(listener);
            resolve();
          }
        };
        chrome.tabs.onUpdated.addListener(listener);
      });
      await wait(5000); 
    } else {
      updateStatus("Page loaded. Initializing scanner...");
    }

    // 2. Get User Name (for filename)
    const userResult = await chrome.scripting.executeScript({
      target: { tabId: startTabId },
      function: getUserName
    });
    if (userResult && userResult[0] && userResult[0].result) {
      userName = userResult[0].result.replace(/[^a-zA-Z0-9]/g, '_'); // Sanitize
    }

    // 3. Scroll the FEED to find ALL playlists
    updateStatus("Scrolling feed to find all playlists...");
    await chrome.scripting.executeScript({
      target: { tabId: startTabId },
      function: scrollFeed
    });

    // 4. Extract Playlist Links with RETRY Logic
    updateStatus("Scanning for playlist links...");
    let playlists = [];
    let attempts = 0;
    
    while (attempts < 3 && playlists.length === 0) {
        attempts++;
        if (attempts > 1) updateStatus(`Retrying scan (Attempt ${attempts}/3)...`);
        
        const result = await chrome.scripting.executeScript({
            target: { tabId: startTabId },
            function: scrapePlaylistLinks
        });
        
        if (result && result[0] && result[0].result) {
            playlists = result[0].result;
        }
        
        if (playlists.length === 0) {
            await wait(2000);
        }
    }
    
    if (playlists.length === 0) {
      throw new Error("No playlists found.\n1. Ensure you are on the 'Playlists' tab.\n2. Refresh the page manually and try again.");
    }

    allPlaylists = playlists;
    updateStatus(`Found ${playlists.length} playlists. Starting video extraction...`);
    
    // 5. Iterate through playlists
    for (let i = 0; i < playlists.length; i++) {
      const p = playlists[i];
      updateStatus(`Processing ${i+1}/${playlists.length}: ${p.title}`);
      
      // Navigate to playlist
      await chrome.tabs.update(startTabId, { url: p.url });
      
      // Wait for navigation
      await new Promise(resolve => {
        const listener = (tabId, info) => {
          if (tabId === startTabId && info.status === 'complete') {
            chrome.tabs.onUpdated.removeListener(listener);
            resolve();
          }
        };
        chrome.tabs.onUpdated.addListener(listener);
      });
      
      // Wait for dynamic content
      await wait(2500);

      // Inject Scroller for VIDEOS
      const videoResult = await chrome.scripting.executeScript({
        target: { tabId: startTabId },
        function: autoScrollAndScrapeVideos
      });
      
      const videos = videoResult[0]?.result || [];
      
      // Add to collection (Raw data first)
      videos.forEach(v => {
        allVideos.push({
          _rawPlaylist: p.title, // Temp storage
          _rawPlaylistURL: p.url,
          ...v
        });
      });
    }

    // 6. Post-Process (Duplicate Detection & Formatting)
    if (allVideos.length > 0) {
      updateStatus("Analyzing for duplicates...");
      
      // Count occurrences of each VideoID
      const videoCounts = {};
      allVideos.forEach(v => {
        if(v.VideoID) {
          videoCounts[v.VideoID] = (videoCounts[v.VideoID] || 0) + 1;
        }
      });

      // Create Clean Data Set with Explicit Order
      const finalData = allVideos.map(v => {
        const isDup = v.VideoID && videoCounts[v.VideoID] > 1;
        return {
          "Playlist Name": v._rawPlaylist,
          "Duplicate": isDup ? "Yes" : "No",
          "Video Title": v.Title,
          "Channel": v.Channel,
          "Video ID": v.VideoID,
          "Position": v.Position,
          "Video URL": v.URL,
          "Playlist URL": v._rawPlaylistURL
        };
      });

      updateStatus(`Extraction Complete! Saving CSV for ${userName}...`);
      const filename = `TubeList_${userName}_${new Date().toISOString().slice(0,10)}.csv`;
      await generateCSV(finalData, filename);
    } else {
      updateStatus("Finished, but found no videos to save.");
    }
    
  } catch (err) {
    updateStatus("Error: " + err.message);
    console.error(err);
  } finally {
    isScraping = false;
    chrome.storage.local.set({ isScraping: false });
  }
}


// === INJECTED FUNCTIONS ===

function getUserName() {
  // Try to find the user's name from the avatar image alt text or top bar
  const avatar = document.querySelector('#avatar-btn img');
  if (avatar && avatar.alt) return avatar.alt;
  
  // Fallback to channel handle if available
  const handle = document.querySelector('#channel-handle');
  if (handle) return handle.innerText;
  
  return "User";
}

async function scrollFeed() {
  return new Promise((resolve) => {
    let lastHeight = 0;
    let noChangeFrames = 0;
    
    const timer = setInterval(() => {
      window.scrollTo(0, document.documentElement.scrollHeight);
      const currentHeight = document.documentElement.scrollHeight;
      
      if (currentHeight === lastHeight) {
        noChangeFrames++;
      } else {
        noChangeFrames = 0;
        lastHeight = currentHeight;
      }

      if (noChangeFrames > 25) { 
        clearInterval(timer);
        window.scrollTo(0, 0); 
        resolve();
      }
    }, 100);
  });
}

function scrapePlaylistLinks() {
  const results = [];
  const seen = new Set();

  const add = (title, url) => {
    if (!title || !url) return;
    try {
        const u = new URL(url);
        const listId = u.searchParams.get('list');
        if (!listId) return;
        
        const cleanUrl = `https://www.youtube.com/playlist?list=${listId}`;
        
        if (!seen.has(cleanUrl)) {
            seen.add(cleanUrl);
            results.push({ title: title.trim(), url: cleanUrl });
        }
    } catch(e) {}
  };

  const items = document.querySelectorAll('ytd-grid-playlist-renderer, ytd-playlist-renderer, ytd-rich-item-renderer, ytd-compact-playlist-renderer');
  items.forEach(item => {
      const titleEl = item.querySelector('#video-title');
      const linkEl = item.querySelector('a[href*="list="]') || item.querySelector('a#thumbnail');
      
      if (titleEl && linkEl) {
          const title = titleEl.innerText || titleEl.getAttribute('title') || titleEl.getAttribute('aria-label');
          if (title) {
            add(title, linkEl.href);
          }
      }
  });

  if (results.length === 0) {
      document.querySelectorAll('a[href*="list="]').forEach(a => {
          if (a.href.includes('/playlist?list=') || a.href.includes('&list=')) {
              let title = a.innerText.trim();
              if (!title) {
                  const parent = a.closest('div');
                  if (parent) {
                      const t = parent.querySelector('#video-title');
                      if (t) title = t.innerText;
                  }
              }
              if (title && title.length > 1) {
                  add(title, a.href);
              }
          }
      });
  }

  return results;
}

async function autoScrollAndScrapeVideos() {
  return new Promise((resolve) => {
    let lastHeight = 0;
    let noChangeFrames = 0;
    
    const timer = setInterval(() => {
      const scrollContainer = document.querySelector('ytd-app'); 
      window.scrollTo(0, document.documentElement.scrollHeight);
      
      const currentHeight = document.documentElement.scrollHeight;
      
      if (currentHeight === lastHeight) {
        noChangeFrames++;
      } else {
        noChangeFrames = 0;
        lastHeight = currentHeight;
      }

      if (noChangeFrames > 40) { 
        clearInterval(timer);
        scrape();
      }
    }, 100);

    function scrape() {
      const data = [];
      const rows = document.querySelectorAll('ytd-playlist-video-renderer');
      
      rows.forEach((row, index) => {
        const titleEl = row.querySelector('#video-title');
        const channelEl = row.querySelector('.ytd-channel-name a'); 
        
        if (titleEl) {
          let videoId = '';
          try {
             videoId = new URL(titleEl.href).searchParams.get('v');
          } catch(e) {}

          data.push({
            Title: titleEl.innerText.trim(),
            VideoID: videoId,
            URL: titleEl.href.split('&')[0], 
            Channel: channelEl ? channelEl.innerText.trim() : 'Unknown',
            Position: index + 1
          });
        }
      });
      resolve(data);
    }
  });
}

function generateCSV(data, filename) {
  return new Promise((resolve, reject) => {
    try {
      const headers = Object.keys(data[0]);
      const csvRows = [];
      
      csvRows.push(headers.join(','));
      
      data.forEach(row => {
        const values = headers.map(header => {
          const val = row[header] === undefined ? '' : '' + row[header];
          const escaped = val.replace(/"/g, '""');
          return `"${escaped}"`;
        });
        csvRows.push(values.join(','));
      });
      
      const csvString = csvRows.join('\n');
      const blob = new Blob([csvString], {type: 'text/csv;charset=utf-8'});
      
      const reader = new FileReader();
      reader.onload = function() {
        const url = this.result;
        chrome.downloads.download({
          url: url,
          filename: filename || 'tubelist_export.csv',
          conflictAction: 'uniquify'
        }, (downloadId) => {
          if (chrome.runtime.lastError) {
             console.error(chrome.runtime.lastError);
             updateStatus("Download Error: " + chrome.runtime.lastError.message);
             reject(chrome.runtime.lastError);
          } else {
             updateStatus("Success! CSV saved to downloads folder.");
             resolve(downloadId);
          }
        });
      };
      reader.readAsDataURL(blob);
      
    } catch (e) {
      console.error(e);
      updateStatus("Error generating CSV: " + e.message);
      reject(e);
    }
  });
}
